﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.CRM.Plugins.Data;
using VA.PPMS.CRM.Plugins.Helper;

namespace VA.PPMS.CRM.Plugins.Agreement
{
    public class AgreementValidation : IPlugin
    {
        private ITracingService _tracingService;
        private IOrganizationService _service;
        private const string PluginName = "SpManagement.AgreementValidation";
        private static string SecureSPCode = "";

        public AgreementValidation(string unsecure, string secure)
        {
            //string xml = "<settings><SPCode>spc</SPCode><StorageAccountKey>sak</StorageAccountKey></settings>";
            SecureSPCode = SettingsHelper.GetValueFromKey(secure, "SPCode");
        }

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                Log("Begin");

                // Obtain the target entity from the input parameters.
                Entity target = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (target.LogicalName != "ppms_provideragreement")
                    return;

                Log("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Get related entities to deactivate
                    Log("Agreement validation begins");

                    var messages = new StringBuilder();

                    // Validate date value is not greater than today
                    DateTime? signedDate = target.GetAttributeValue<DateTime?>("ppms_medicaldirectorsignaturedate");
                    OptionSetValue agreementType = target.GetAttributeValue<OptionSetValue>("ppms_agreementtype");

                    Log($"Signed Date: {signedDate.ToString()}");
                    if (!signedDate.HasValue)
                    {
                        messages.AppendLine("Medical Director Signature Date is not provided.");
                    }
                    else if (signedDate.Value > DateTime.Today.AddDays(1))
                    {
                        messages.AppendLine("Medical Director Signature Date cannot be in the future.");
                    }

                    // Retrieve settings
                    var settings = SettingsHelper.GetSettings(_service);
                    if (!settings.IsValid)
                    {
                        var errorMessage = $"ERROR: Unable to retrieve service settings.\nPlease contact the system administrator.\n";
                        Log("ERROR: Unable to retrieve service settings. {0}", settings.ErrorMessage);
                        return;
                    }
                    settings.SpCode = SecureSPCode;

                    // Retrieve entity details
                    var entity = GetAgreement(target);

                    // Do not do a file check for IA types
                    if (agreementType == null) agreementType = entity.GetAttributeValue<OptionSetValue>("ppms_agreementtype");
                    if (agreementType != null && agreementType.Value == 767940000) return;

                    // Validate SharePoint folder exists
                    if (entity != null && !HasFilesInSharePoint(entity, settings))
                    {
                        messages.AppendLine("Attachment has not been uploaded to Documents folder.");
                    }

                    if (messages.Length > 0)
                    {
                        var errorMessage = $"Validation error:\n\n{messages.ToString()}\n";
                        Log(errorMessage);
                        throw new InvalidPluginExecutionException(errorMessage);
                    }

                    Log("Processing complete.");
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    Log("Fault: {0}", ex.Message);
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    Log("Exception: {0}", ex.Message);
                    throw;
                }
            }
            Log("Done");
        }

        /// <summary>
        /// Determines if the specified entity has uploaded files associated with it
        /// </summary>
        /// <param name="entity">Agreement entity</param>
        /// <returns>true if files exist</returns>
        private bool HasFilesInSharePoint(Entity entity, PpmsServiceSettings settings)
        {
            bool hasFiles = false;

            try
            {
                string baseUrl = settings.SpBaseUrl; // "https://dvagov.sharepoint.com/sites/ppms/dev";
                string library = settings.SpLibrary; // "ppms_provideragreement";
                string folderName = GetFolderName(entity);
                string folderPath = $"{library}/{folderName}";

                Log($"Checking for files: {folderPath}");

                var token = GetSecurityToken(settings);

                // Lookup folder contents listing
                var result = SharePointHelper.GetFolderContents(token, baseUrl, folderPath);

                // Convert JSON to object
                var spFiles = JsonHelper.Deserialize<SpFolderContents>(result);
                var count = spFiles.d.results.Count();
                Log($"File count: {count}");

                // Check file count
                if (count > 0)
                {
                    hasFiles = true;
                }
            }
            catch (Exception ex)
            {
                Log($"ERROR: {ex.Message}");
            }

            return hasFiles;
        }

        private string GetSecurityToken(PpmsServiceSettings settings)
        {
            string resourceId = "00000003-0000-0ff1-ce00-000000000000";
            string spUrl = $"https://{settings.SpDomain}/_vti_bin/client.svc/";

            var tenantId = SharePointHelper.GetTenantId(spUrl);
            return SharePointHelper.GetAuthorisationToken(settings.SpDomain, tenantId, resourceId, settings.SpId, settings.SpCode);
        }

        /// <summary>
        /// Build a folder name based on the Agreement information
        /// </summary>
        /// <param name="agreement">Provider agreement entity</param>
        /// <returns></returns>
        private string GetFolderName(Entity agreement)
        {
            if (agreement == null) return string.Empty;

            FilterExpression filter = new FilterExpression();
            filter.AddCondition("regardingobjectid", ConditionOperator.Equal, agreement.Id);

            QueryExpression query = new QueryExpression("sharepointdocumentlocation");
            query.ColumnSet.AddColumns("sitecollectionid", "name", "relativeurl");
            query.Criteria.AddFilter(filter);

            var results = _service.RetrieveMultiple(query);

            return results != null && results.Entities.Count > 0 ? results[0].GetAttributeValue<string>("relativeurl") : null;
        }

        private Entity GetAgreement(Entity entity)
        {
            if (entity != null)
            {
                return _service.Retrieve("ppms_provideragreement", entity.Id, new ColumnSet(new string[] { "ppms_provideragreementid", "ppms_agreementid", "ppms_agreementtype" }));
            }

            return null;
        }

        private void Log(string message, params object[] args)
        {
            _tracingService.Trace(message, args);
        }
    }
}
